<?php
// =============================================
// INITIALIZATION & SECURITY
// =============================================

// Basic security check
if (!defined('ACCESS_ALLOWED')) {
    define('ACCESS_ALLOWED', true);
}

// Error reporting configuration
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
ini_set('log_errors', 1);

// =============================================
// PERFORMANCE CONFIGURATION
// =============================================

// Increase memory and execution time for large datasets
ini_set('memory_limit', '512M');
ini_set('max_execution_time', 120);
ini_set('max_input_vars', 3000);

// =============================================
// FILE SYSTEM CONFIGURATION
// =============================================

// Directory and file paths
$baseDir = dirname(__DIR__);
$logDir = $baseDir . '/logs';
$logFile = $logDir . '/tracking.log';
$errorLogFile = $logDir . '/tracking_errors.log';
ini_set('error_log', $errorLogFile);

// Create logs directory if it doesn't exist
if (!file_exists($logDir)) {
    if (!mkdir($logDir, 0755, true)) {
        error_log("Failed to create logs directory: $logDir");
        die("System configuration error. Please contact administrator.");
    }
}

// Verify log file exists and is writable
if (!file_exists($logFile)) {
    if (!touch($logFile)) {
        error_log("Failed to create log file: $logFile");
        die("System configuration error. Please contact administrator.");
    }
}
if (!is_writable($logFile)) {
    error_log("Log file not writable: $logFile");
    die("System configuration error. Please contact administrator.");
}

// =============================================
// COUNTRY DETECTION CONFIGURATION
// =============================================

// Country flag mapping (ISO 3166-1 alpha-2 codes)
$countryFlags = [
    'US' => '🇺🇸', 'GB' => '🇬🇧', 'CA' => '🇨🇦', 'AU' => '🇦🇺', 'DE' => '🇩🇪',
    'FR' => '🇫🇷', 'IT' => '🇮🇹', 'ES' => '🇪🇸', 'JP' => '🇯🇵', 'CN' => '🇨🇳',
    'IN' => '🇮🇳', 'BR' => '🇧🇷', 'RU' => '🇷🇺', 'MA' => '🇲🇦', 'SA' => '🇸🇦',
    'AE' => '🇦🇪', 'EG' => '🇪🇬', 'ZA' => '🇿🇦', 'NG' => '🇳🇬', 'KE' => '🇰🇪',
    'LOCAL' => '🏠', 'UN' => '🌐'
];

// Local IP to country mapping for fallback
$ipCountryMap = [
    '127.0.0.1' => 'LOCAL',
    '::1' => 'LOCAL'
];

// =============================================
// FUNCTIONS
// =============================================

/**
 * Get country information from IP address
 */
function getCountryFromIP($ip, $ipCountryMap, $countryFlags) {
    // Check local mapping first
    if (isset($ipCountryMap[$ip])) {
        return [
            'country_code' => $ipCountryMap[$ip],
            'country_name' => $ip === '127.0.0.1' ? 'Localhost' : 'Private Network',
            'country_flag' => $countryFlags[$ipCountryMap[$ip]] ?? $countryFlags['UN']
        ];
    }
    
    // Try freegeoip.app with timeout
    $url = "https://freegeoip.app/json/$ip";
    $options = [
        'http' => [
            'timeout' => 2, // 2 second timeout
            'ignore_errors' => true
        ]
    ];
    
    try {
        $context = stream_context_create($options);
        $response = @file_get_contents($url, false, $context);
        
        if ($response !== false) {
            $data = json_decode($response, true);
            if (isset($data['country_code'])) {
                return [
                    'country_code' => $data['country_code'],
                    'country_name' => $data['country_name'] ?? $data['country_code'],
                    'country_flag' => $countryFlags[$data['country_code']] ?? $countryFlags['UN']
                ];
            }
        }
    } catch (Exception $e) {
        error_log("GeoIP error for IP $ip: " . $e->getMessage());
    }
    
    return [
        'country_code' => 'UN',
        'country_name' => 'Unknown',
        'country_flag' => $countryFlags['UN']
    ];
}

/**
 * Parse log entry into structured data
 */
function parseLogEntry($entry) {
    // Improved parsing to handle the format: Time|IP|Email|Country|User Agent
    $parts = explode('|', $entry);
    
    // Clean each part and provide defaults
    $time = trim($parts[0] ?? 'Unknown Time');
    $ip = trim($parts[1] ?? 'Unknown IP');
    $email = trim($parts[2] ?? '');
    $country = trim($parts[3] ?? 'Unknown Country');
    $ua = trim($parts[4] ?? 'Unknown User Agent');
    
    return [
        'time' => $time,
        'ip' => $ip,
        'email' => $email === '' ? null : $email,
        'country_raw' => $country,
        'ua' => $ua,
        'raw' => $entry
    ];
}

/**
 * Extract email from user agent string if present (fallback)
 */
function extractEmailFromUA($ua) {
    if (preg_match('/\[email:(.*?)\]/', $ua, $matches)) {
        return trim($matches[1]);
    }
    return null;
}

/**
 * Efficiently read large log files with pagination
 */
function readLogTail($logFile, $limit = null, $offset = 0) {
    if (!file_exists($logFile) || !is_readable($logFile)) {
        return [];
    }
    
    $entries = [];
    $file = fopen($logFile, 'r');
    
    if (!$file) {
        return [];
    }
    
    // If no limit, read all entries (for small files)
    if ($limit === null) {
        while (($line = fgets($file)) !== false) {
            $entries[] = trim($line);
        }
        $entries = array_reverse($entries);
        fclose($file);
        return $entries;
    }
    
    // For large files with pagination, read efficiently
    $lines = [];
    $currentLine = '';
    $lineCount = 0;
    $bytesToRead = min(1024 * 1024, filesize($logFile)); // Read 1MB chunks
    
    // Start from the end of file
    fseek($file, 0, SEEK_END);
    $pos = ftell($file);
    
    while ($pos > 0 && count($lines) < $limit) {
        $readSize = min($bytesToRead, $pos);
        fseek($file, $pos - $readSize);
        $chunk = fread($file, $readSize);
        $pos -= $readSize;
        
        $linesInChunk = explode("\n", $chunk);
        
        // Reverse the lines in this chunk
        $linesInChunk = array_reverse($linesInChunk);
        
        foreach ($linesInChunk as $linePart) {
            $currentLine = $linePart . $currentLine;
            
            if (strpos($chunk, "\n") !== false || $pos == 0) {
                if (!empty(trim($currentLine))) {
                    $lineCount++;
                    if ($lineCount > $offset && count($lines) < $limit) {
                        $lines[] = trim($currentLine);
                    }
                }
                $currentLine = '';
            }
        }
    }
    
    fclose($file);
    return array_reverse($lines); // Return in chronological order
}

// =============================================
// MAIN PROCESSING
// =============================================

// Initialize variables
$logEntries = [];
$filteredEntries = [];
$totalOpens = 0;
$uniqueOpens = 0;
$uniqueEmails = 0;
$exportData = false;
$errorMessage = '';
$deleteRequested = false;

// Pagination configuration
$entriesPerPage = 100; // Reduced for better performance
$currentPage = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$offset = ($currentPage - 1) * $entriesPerPage;

// Handle delete request
if (isset($_GET['delete'])) {
    $deleteRequested = true;
    if (file_exists($logFile)) {
        if (unlink($logFile)) {
            touch($logFile);
            header("Location: ?");
            exit();
        } else {
            $errorMessage = "Failed to delete log file.";
        }
    }
}

// Handle export before any output
if (isset($_GET['export'])) {
    $exportData = true;
    
    // Set headers for download
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="email_tracking_export_' . date('Y-m-d') . '.csv"');
}

// Get filter parameters safely
$filterCountry = $_GET['country'] ?? '';
$filterIp = $_GET['ip'] ?? '';
$filterDate = $_GET['date'] ?? '';
$filterEmail = $_GET['email'] ?? '';

// Count total entries efficiently
try {
    if (file_exists($logFile) && is_readable($logFile)) {
        // Count total lines efficiently for large files
        if (filesize($logFile) > 10485760) { // 10MB
            // For large files, estimate line count
            $file = fopen($logFile, 'r');
            $sampleSize = min(1024 * 1024, filesize($logFile)); // Sample 1MB
            fseek($file, 0);
            $sample = fread($file, $sampleSize);
            $linesInSample = substr_count($sample, "\n");
            $totalOpens = ceil($linesInSample * (filesize($logFile) / $sampleSize));
            fclose($file);
        } else {
            // For small files, count exactly
            $fileContent = file_get_contents($logFile);
            $totalOpens = $fileContent ? substr_count($fileContent, "\n") : 0;
        }
        
        // Get unique counts efficiently using sampling for large datasets
        if ($totalOpens > 10000) {
            // For very large datasets, use sampling
            $sampleSize = min(5000, $totalOpens);
            $sampleEntries = readLogTail($logFile, $sampleSize);
            
            $ipCounts = [];
            $emailCounts = [];
            foreach ($sampleEntries as $entry) {
                $parsed = parseLogEntry($entry);
                $ip = $parsed['ip'];
                if (!isset($ipCounts[$ip])) {
                    $ipCounts[$ip] = 0;
                }
                $ipCounts[$ip]++;
                
                $email = $parsed['email'] ?? extractEmailFromUA($parsed['ua']);
                if ($email) {
                    if (!isset($emailCounts[$email])) {
                        $emailCounts[$email] = 0;
                    }
                    $emailCounts[$email]++;
                }
            }
            
            // Estimate unique counts based on sample
            $uniqueOpens = round(count($ipCounts) * ($totalOpens / $sampleSize));
            $uniqueEmails = round(count($emailCounts) * ($totalOpens / $sampleSize));
        } else {
            // For smaller datasets, count exactly
            $allEntries = readLogTail($logFile);
            
            $ipCounts = [];
            $emailCounts = [];
            foreach ($allEntries as $entry) {
                $parsed = parseLogEntry($entry);
                $ip = $parsed['ip'];
                if (!isset($ipCounts[$ip])) {
                    $ipCounts[$ip] = 0;
                }
                $ipCounts[$ip]++;
                
                $email = $parsed['email'] ?? extractEmailFromUA($parsed['ua']);
                if ($email) {
                    if (!isset($emailCounts[$email])) {
                        $emailCounts[$email] = 0;
                    }
                    $emailCounts[$email]++;
                }
            }
            $uniqueOpens = count($ipCounts);
            $uniqueEmails = count($emailCounts);
        }
    } else {
        throw new Exception("Log file not accessible");
    }
} catch (Exception $e) {
    $errorMessage = "Error loading log data: " . $e->getMessage();
    error_log($errorMessage);
    $allEntries = [];
}

// Read only the entries needed for current page
$displayEntries = readLogTail($logFile, $entriesPerPage, $offset);

// Process log entries with error handling (only for displayed entries)
foreach ($displayEntries as $entry) {
    try {
        $parsed = parseLogEntry($entry);
        
        // Get country data - first try from log entry, then fallback to IP lookup
        if ($parsed['country_raw'] !== 'Unknown Country' && $parsed['country_raw'] !== '') {
            $countryCode = $parsed['country_raw'];
            $countryData = [
                'country_code' => $countryCode,
                'country_name' => $countryCode === 'LOCAL' ? 'Localhost' : ($countryCode === 'UN' ? 'Unknown' : $countryCode),
                'country_flag' => $countryFlags[$countryCode] ?? $countryFlags['UN']
            ];
        } else {
            $countryData = getCountryFromIP($parsed['ip'], $ipCountryMap, $countryFlags);
        }
        
        // Get email - first from dedicated field, then fallback to UA extraction
        $email = $parsed['email'] ?? extractEmailFromUA($parsed['ua']);
        
        $logEntry = array_merge($parsed, $countryData, [
            'email' => $email,
            'email_domain' => $email ? substr(strrchr($email, '@'), 1) : null
        ]);
        
        // Apply filters
        $matches = true;
        
        if ($filterCountry && $logEntry['country_code'] !== $filterCountry) {
            $matches = false;
        }
        
        if ($filterIp && strpos($logEntry['ip'], $filterIp) === false) {
            $matches = false;
        }
        
        if ($filterDate && strpos($logEntry['time'], $filterDate) === false) {
            $matches = false;
        }
        
        if ($filterEmail && (!$email || stripos($email, $filterEmail) === false)) {
            $matches = false;
        }
        
        if ($matches) {
            $logEntries[] = $logEntry;
        }
    } catch (Exception $e) {
        error_log("Error processing log entry: " . $e->getMessage());
        continue;
    }
}

// Calculate statistics for dashboard (using sampled data for large datasets)
$countries = [];
$emailDomains = [];
$timelineData = [];
$hourlyData = array_fill(0, 24, 0);

// Use sampling for statistics with large datasets
$statSampleSize = min(2000, $totalOpens);
$statSampleEntries = readLogTail($logFile, $statSampleSize);

foreach ($statSampleEntries as $entry) {
    try {
        $parsed = parseLogEntry($entry);
        
        // Country stats
        $countryCode = $parsed['country_raw'] !== 'Unknown Country' && $parsed['country_raw'] !== '' 
            ? $parsed['country_raw'] 
            : 'UN';
        
        if (!isset($countries[$countryCode])) {
            $countries[$countryCode] = [
                'count' => 0,
                'name' => $countryCode === 'LOCAL' ? 'Localhost' : ($countryCode === 'UN' ? 'Unknown' : $countryCode),
                'flag' => $countryFlags[$countryCode] ?? $countryFlags['UN']
            ];
        }
        $countries[$countryCode]['count']++;
        
        // Email domain stats
        $email = $parsed['email'] ?? extractEmailFromUA($parsed['ua']);
        if ($email) {
            $domain = substr(strrchr($email, '@'), 1);
            if ($domain) {
                if (!isset($emailDomains[$domain])) {
                    $emailDomains[$domain] = 0;
                }
                $emailDomains[$domain]++;
            }
        }
        
        // Timeline data (by hour)
        $hour = (int)date('H', strtotime($parsed['time']));
        $hourlyData[$hour]++;
    } catch (Exception $e) {
        continue;
    }
}

// Scale counts for sampled data
if ($totalOpens > $statSampleSize) {
    $scaleFactor = $totalOpens / $statSampleSize;
    foreach ($countries as $code => $data) {
        $countries[$code]['count'] = round($data['count'] * $scaleFactor);
    }
    foreach ($emailDomains as $domain => $count) {
        $emailDomains[$domain] = round($count * $scaleFactor);
    }
    foreach ($hourlyData as $hour => $count) {
        $hourlyData[$hour] = round($count * $scaleFactor);
    }
}

// Sort countries by count
uasort($countries, function($a, $b) {
    return $b['count'] - $a['count'];
});

// Sort email domains by count
arsort($emailDomains);

// Prepare timeline data for chart
$timelineLabels = [];
$timelineValues = [];
foreach ($hourlyData as $hour => $count) {
    $timelineLabels[] = sprintf("%02d:00", $hour);
    $timelineValues[] = $count;
}

// Get top 5 countries and domains
$topCountries = array_slice($countries, 0, 5, true);
$topDomains = array_slice($emailDomains, 0, 5, true);

// Calculate pagination
$totalPages = ceil($totalOpens / $entriesPerPage);

// Handle CSV export
if ($exportData) {
    try {
        $output = fopen('php://output', 'w');
        
        // Write CSV headers
        fputcsv($output, ['Timestamp', 'IP Address', 'Country Code', 'Country Name', 'Email', 'Email Domain', 'User Agent']);
        
        // Export in chunks to avoid memory issues
        $exportChunkSize = 1000;
        $exportOffset = 0;
        
        do {
            $exportEntries = readLogTail($logFile, $exportChunkSize, $exportOffset);
            
            foreach ($exportEntries as $entry) {
                try {
                    $parsed = parseLogEntry($entry);
                    
                    // Get country data
                    if ($parsed['country_raw'] !== 'Unknown Country' && $parsed['country_raw'] !== '') {
                        $countryCode = $parsed['country_raw'];
                        $countryName = $countryCode === 'LOCAL' ? 'Localhost' : ($countryCode === 'UN' ? 'Unknown' : $countryCode);
                    } else {
                        $countryData = getCountryFromIP($parsed['ip'], $ipCountryMap, $countryFlags);
                        $countryCode = $countryData['country_code'];
                        $countryName = $countryData['country_name'];
                    }
                    
                    $email = $parsed['email'] ?? extractEmailFromUA($parsed['ua']);
                    
                    fputcsv($output, [
                        $parsed['time'],
                        $parsed['ip'],
                        $countryCode,
                        $countryName,
                        $email ?? '',
                        $email ? substr(strrchr($email, '@'), 1) : '',
                        $parsed['ua']
                    ]);
                } catch (Exception $e) {
                    continue;
                }
            }
            
            $exportOffset += $exportChunkSize;
            flush(); // Flush output buffer to prevent timeout
        } while (!empty($exportEntries));
        
        fclose($output);
        exit();
    } catch (Exception $e) {
        error_log("Export failed: " . $e->getMessage());
        die("An error occurred during export. Please try again.");
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>📊 Email Tracker Dashboard | WebTigersAI</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/chart.js@3.7.1/dist/chart.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        :root {
            --primary: #4a6baf;
            --primary-light: #6a8ad8;
            --secondary: #4CAF50;
            --danger: #e74c3c;
            --warning: #f39c12;
            --info: #3498db;
            --dark: #2c3e50;
            --light: #f8f9fa;
            --gray: #95a5a6;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f7fa;
            margin: 0;
            padding: 0;
            color: #333;
            line-height: 1.6;
        }
        
        header {
            background: linear-gradient(135deg, var(--primary), var(--primary-light));
            color: white;
            padding: 1.5rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .logo {
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 1.5rem;
            font-weight: 600;
        }
        
        .logo i {
            font-size: 1.8rem;
        }
        
        .last-updated {
            font-size: 0.9rem;
            opacity: 0.9;
        }
        
        .container {
            margin: 2rem auto;
            padding: 0 2rem;
            max-width: 1800px;
        }
        
        .dashboard-grid {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .card {
            background: white;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
            padding: 20px;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 15px rgba(0, 0, 0, 0.1);
        }
        
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }
        
        .card-title {
            font-size: 1rem;
            color: var(--gray);
            font-weight: 600;
            margin: 0;
        }
        
        .card-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }
        
        .card-icon.total {
            background: var(--info);
        }
        
        .card-icon.unique {
            background: var(--secondary);
        }
        
        .card-icon.rate {
            background: var(--warning);
        }
        
        .card-icon.recent {
            background: var(--danger);
        }
        
        .card-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--dark);
            margin: 5px 0;
        }
        
        .card-change {
            font-size: 0.85rem;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .card-change.positive {
            color: var(--secondary);
        }
        
        .card-change.negative {
            color: var(--danger);
        }
        
        .main-content {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }
        
        .log-section {
            background: white;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
            padding: 25px;
            grid-column: 1 / -1;
        }
        
        .stats-section {
            background: white;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
            padding: 25px;
        }
        
        .chart-section {
            background: white;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
            padding: 25px;
        }
        
        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .section-title {
            font-size: 1.5rem;
            color: var(--dark);
            margin: 0;
        }
        
        .actions {
            display: flex;
            gap: 10px;
        }
        
        .btn {
            padding: 8px 15px;
            border-radius: 5px;
            border: none;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 0.9rem;
            transition: all 0.2s ease;
        }
        
        .btn-primary {
            background: var(--primary);
            color: white;
        }
        
        .btn-primary:hover {
            background: var(--primary-light);
        }
        
        .btn-secondary {
            background: var(--light);
            color: var(--dark);
            border: 1px solid #ddd;
        }
        
        .btn-secondary:hover {
            background: #e9ecef;
        }
        
        .btn-success {
            background: var(--secondary);
            color: white;
        }
        
        .btn-success:hover {
            background: #43a047;
        }
        
        .btn-danger {
            background: var(--danger);
            color: white;
        }
        
        .btn-danger:hover {
            background: #c0392b;
        }
        
        .log-container {
            position: relative;
        }
        
        .log-view {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 8px;
            max-height: 500px;
            overflow-y: auto;
            font-family: 'Courier New', Courier, monospace;
            font-size: 0.85rem;
            border-left: 4px solid var(--primary);
        }
        
        .log-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .log-table th {
            background-color: var(--primary);
            color: white;
            padding: 10px;
            text-align: left;
            position: sticky;
            top: 0;
        }
        
        .log-table td {
            padding: 10px;
            border-bottom: 1px solid #eee;
        }
        
        .log-table tr:hover {
            background-color: rgba(74, 107, 175, 0.1);
        }
        
        .log-time {
            color: var(--primary);
            font-weight: 600;
            white-space: nowrap;
        }
        
        .log-ip {
            color: var(--secondary);
            white-space: nowrap;
        }
        
        .log-country {
            font-size: 1.2rem;
            text-align: center;
            white-space: nowrap;
        }
        
        .log-email {
            color: var(--info);
            white-space: nowrap;
        }
        
        .log-useragent {
            color: var(--gray);
            font-size: 0.8rem;
        }
        
        .footer {
            text-align: center;
            font-size: 0.85rem;
            color: var(--gray);
            margin: 3rem 0 2rem;
            padding-top: 1rem;
            border-top: 1px solid #eee;
        }
        
        /* Filter form styles */
        .filter-form {
            background: white;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
            padding: 20px;
            margin-bottom: 20px;
            display: none;
        }
        
        .filter-form.active {
            display: block;
            animation: fadeIn 0.3s ease-in-out;
        }
        
        .filter-row {
            display: flex;
            gap: 15px;
            margin-bottom: 15px;
            flex-wrap: wrap;
        }
        
        .filter-group {
            flex: 1;
            min-width: 200px;
        }
        
        .filter-label {
            display: block;
            margin-bottom: 5px;
            font-weight: 600;
            font-size: 0.9rem;
        }
        
        .filter-input {
            width: 100%;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-family: inherit;
        }
        
        .filter-actions {
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        
        /* Error message styling */
        .error-message {
            background: #fee;
            border-left: 4px solid var(--danger);
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 4px;
            color: var(--danger);
            animation: fadeIn 0.5s ease-in-out;
        }
        
        /* Auto-refresh indicator */
        .auto-refresh {
            display: inline-flex;
            align-items: center;
            gap: 5px;
            font-size: 0.8rem;
            color: var(--gray);
            margin-left: 15px;
        }
        
        .pulse {
            width: 10px;
            height: 10px;
            border-radius: 50%;
            background: var(--secondary);
            animation: pulse 2s infinite;
        }
        
        /* Stats list styles */
        .stats-list {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        
        .stats-item {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid #eee;
            animation: fadeIn 0.5s ease-in-out;
        }
        
        .stats-item:last-child {
            border-bottom: none;
        }
        
        .stats-label {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .stats-value {
            font-weight: 600;
        }
        
        /* Chart container */
        .chart-container {
            position: relative;
            height: 300px;
            width: 100%;
        }
        
        /* Pagination styles */
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            margin: 20px 0;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        .pagination .btn {
            padding: 8px 15px;
            text-decoration: none;
        }
        
        .pagination-info {
            margin: 0 15px;
            font-weight: 600;
            color: var(--dark);
        }
        
        /* Animations */
        @keyframes pulse {
            0% { opacity: 0.6; transform: scale(0.95); }
            50% { opacity: 1; transform: scale(1.05); }
            100% { opacity: 0.6; transform: scale(0.95); }
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .animate-fade-in {
            animation: fadeIn 0.5s ease-in-out;
        }
        
        /* Responsive adjustments */
        @media (max-width: 1200px) {
            .dashboard-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .main-content {
                grid-template-columns: 1fr;
            }
        }
        
        @media (max-width: 768px) {
            header {
                flex-direction: column;
                gap: 10px;
                text-align: center;
            }
            
            .dashboard-grid {
                grid-template-columns: 1fr;
            }
            
            .actions {
                flex-wrap: wrap;
                justify-content: flex-end;
            }
            
            .pagination {
                flex-direction: column;
            }
        }
        
        /* Tooltip styles */
        .tooltip {
            position: relative;
            display: inline-block;
        }
        
        .tooltip .tooltiptext {
            visibility: hidden;
            width: 200px;
            background-color: #555;
            color: #fff;
            text-align: center;
            border-radius: 6px;
            padding: 5px;
            position: absolute;
            z-index: 1;
            bottom: 125%;
            left: 50%;
            margin-left: -100px;
            opacity: 0;
            transition: opacity 0.3s;
        }
        
        .tooltip:hover .tooltiptext {
            visibility: visible;
            opacity: 1;
        }
        
        /* Loading indicator */
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid var(--primary);
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <header>
        <div class="logo">
            <i class="fas fa-envelope-open-text"></i>
            <span>Email Tracker By : Fikri</span>
        </div>
        <div class="last-updated">
            Last updated: <?php echo date('Y-m-d H:i:s'); ?>
            <?php if(!$filterCountry && !$filterIp && !$filterDate && !$filterEmail && $currentPage === 1): ?>
            <span class="auto-refresh">
                <span class="pulse"></span>
                Auto-refresh in 30s
            </span>
            <?php endif; ?>
        </div>
    </header>
    
    <div class="container">
        <?php if (!empty($errorMessage)): ?>
            <div class="error-message">
                <i class="fas fa-exclamation-circle"></i>
                <?php echo htmlspecialchars($errorMessage); ?>
            </div>
        <?php endif; ?>
        
        <div class="dashboard-grid animate-fade-in">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Total Opens</h3>
                    <div class="card-icon total">
                        <i class="fas fa-envelope"></i>
                    </div>
                </div>
                <div class="card-value"><?php echo number_format($totalOpens); ?></div>
                <div class="card-change positive">
                    <i class="fas fa-arrow-up"></i>
                    <span>Live tracking</span>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Unique Opens</h3>
                    <div class="card-icon unique">
                        <i class="fas fa-user-check"></i>
                    </div>
                </div>
                <div class="card-value"><?php echo number_format($uniqueOpens); ?></div>
                <div class="card-change positive">
                    <i class="fas fa-arrow-up"></i>
                    <span>Unique visitors</span>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Unique Emails</h3>
                    <div class="card-icon rate">
                        <i class="fas fa-at"></i>
                    </div>
                </div>
                <div class="card-value"><?php echo number_format($uniqueEmails ?? 0); ?></div>
                <div class="card-change positive">
                    <i class="fas fa-arrow-up"></i>
                    <span>Tracked emails</span>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Recent Activity</h3>
                    <div class="card-icon recent">
                        <i class="fas fa-clock"></i>
                    </div>
                </div>
                <div class="card-value"><?php echo $totalOpens > 0 ? 'Active' : 'None'; ?></div>
                <div class="card-change">
                    <i class="fas fa-circle"></i>
                    <span>Real-time</span>
                </div>
            </div>
        </div>
        
        <!-- Filter Form -->
        <div class="filter-form" id="filterForm">
            <form method="GET" action="">
                <input type="hidden" name="page" value="1">
                <div class="filter-row">
                    <div class="filter-group">
                        <label class="filter-label">Country</label>
                        <select name="country" class="filter-input">
                            <option value="">All Countries</option>
                            <?php foreach($countryFlags as $code => $flag): ?>
                                <?php if($code !== 'LOCAL' && $code !== 'UN'): ?>
                                    <option value="<?php echo htmlspecialchars($code); ?>" <?php echo $filterCountry === $code ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($flag . ' ' . $code); ?>
                                    </option>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="filter-group">
                        <label class="filter-label">IP Address</label>
                        <input type="text" name="ip" class="filter-input" placeholder="Filter by IP" value="<?php echo htmlspecialchars($filterIp); ?>">
                    </div>
                    <div class="filter-group">
                        <label class="filter-label">Date</label>
                        <input type="date" name="date" class="filter-input" value="<?php echo htmlspecialchars($filterDate); ?>">
                    </div>
                    <div class="filter-group">
                        <label class="filter-label">Email</label>
                        <input type="text" name="email" class="filter-input" placeholder="Filter by email" value="<?php echo htmlspecialchars($filterEmail); ?>">
                    </div>
                </div>
                <div class="filter-actions">
                    <button type="button" class="btn btn-secondary" onclick="document.getElementById('filterForm').classList.remove('active')">
                        Cancel
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-filter"></i> Apply Filters
                    </button>
                    <a href="?" class="btn btn-danger">
                        <i class="fas fa-times"></i> Clear All
                    </a>
                </div>
            </form>
        </div>
        
        <div class="main-content">
            <div class="stats-section animate-fade-in">
                <div class="section-header">
                    <h2 class="section-title">Statistics</h2>
                </div>
                
                <div class="chart-container">
                    <canvas id="timelineChart"></canvas>
                </div>
                
                <h3>Top Countries</h3>
                <ul class="stats-list">
                    <?php foreach($topCountries as $code => $country): ?>
                        <li class="stats-item">
                            <span class="stats-label">
                                <?php echo htmlspecialchars($country['flag'] . ' ' . $country['name']); ?>
                            </span>
                            <span class="stats-value"><?php echo number_format($country['count']); ?></span>
                        </li>
                    <?php endforeach; ?>
                </ul>
                
                <?php if(!empty($topDomains)): ?>
                    <h3>Top Email Domains</h3>
                    <ul class="stats-list">
                        <?php foreach($topDomains as $domain => $count): ?>
                            <li class="stats-item">
                                <span class="stats-label">
                                    <i class="fas fa-at"></i> <?php echo htmlspecialchars($domain); ?>
                                </span>
                                <span class="stats-value"><?php echo number_format($count); ?></span>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
            
            <div class="chart-section animate-fade-in">
                <div class="section-header">
                    <h2 class="section-title">Activity Map</h2>
                </div>
                <div class="chart-container">
                    <canvas id="countryChart"></canvas>
                </div>
            </div>
            
            <div class="log-section animate-fade-in">
                <div class="section-header">
                    <h2 class="section-title">Tracking Logs</h2>
                    <div class="actions">
                        <button class="btn btn-secondary" onclick="document.getElementById('filterForm').classList.toggle('active')">
                            <i class="fas fa-filter"></i> Filter
                        </button>
                        <a href="?export=1<?php 
                            echo $filterCountry ? '&country='.htmlspecialchars($filterCountry) : ''; 
                            echo $filterIp ? '&ip='.htmlspecialchars($filterIp) : ''; 
                            echo $filterDate ? '&date='.htmlspecialchars($filterDate) : ''; 
                            echo $filterEmail ? '&email='.htmlspecialchars($filterEmail) : ''; 
                        ?>" class="btn btn-success">
                            <i class="fas fa-download"></i> Export
                        </a>
                        <a href="?delete=1" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete all logs? This cannot be undone.');">
                            <i class="fas fa-trash"></i> Delete Logs
                        </a>
                    </div>
                </div>
                
                <div class="log-container">
                    <div class="log-view">
                        <?php if(count($logEntries) > 0): ?>
                            <table class="log-table">
                                <thead>
                                    <tr>
                                        <th>Time</th>
                                        <th>IP Address</th>
                                        <th>Country</th>
                                        <th>Email</th>
                                        <th>User Agent</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach($logEntries as $entry): ?>
                                        <tr class="animate-fade-in">
                                            <td class="log-time"><?php echo htmlspecialchars($entry['time']); ?></td>
                                            <td class="log-ip"><?php echo htmlspecialchars($entry['ip']); ?></td>
                                            <td class="log-country tooltip" title="<?php echo htmlspecialchars($entry['country_name']); ?>">
                                                <?php echo htmlspecialchars($entry['country_flag']); ?>
                                            </td>
                                            <td class="log-email">
                                                <?php if($entry['email']): ?>
                                                    <span class="tooltip" title="Domain: <?php echo htmlspecialchars($entry['email_domain'] ?? 'Unknown'); ?>">
                                                        <?php echo htmlspecialchars($entry['email']); ?>
                                                    </span>
                                                <?php else: ?>
                                                    <span class="tooltip" title="No email detected in tracking pixel">
                                                        <i class="fas fa-eye"></i> No email
                                                    </span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="log-useragent tooltip" title="<?php echo htmlspecialchars($entry['ua']); ?>">
                                                <?php echo htmlspecialchars(substr($entry['ua'], 0, 50) . (strlen($entry['ua']) > 50 ? '...' : '')); ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php else: ?>
                            <div class="log-entry">
                                <?php if($filterCountry || $filterIp || $filterDate || $filterEmail): ?>
                                    No tracking data found matching your filters.
                                <?php else: ?>
                                    No tracking data available yet.
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <?php if ($totalPages > 1): ?>
                <div class="pagination">
                    <?php if ($currentPage > 1): ?>
                        <a href="?page=<?php echo $currentPage - 1; ?><?php 
                            echo $filterCountry ? '&country='.htmlspecialchars($filterCountry) : ''; 
                            echo $filterIp ? '&ip='.htmlspecialchars($filterIp) : ''; 
                            echo $filterDate ? '&date='.htmlspecialchars($filterDate) : ''; 
                            echo $filterEmail ? '&email='.htmlspecialchars($filterEmail) : ''; 
                        ?>" class="btn btn-secondary">
                            <i class="fas fa-chevron-left"></i> Previous
                        </a>
                    <?php endif; ?>
                    
                    <span class="pagination-info">
                        Page <?php echo $currentPage; ?> of <?php echo $totalPages; ?>
                        (Showing <?php echo count($logEntries); ?> of ~<?php echo number_format($totalOpens); ?> entries)
                    </span>
                    
                    <?php if ($currentPage < $totalPages): ?>
                        <a href="?page=<?php echo $currentPage + 1; ?><?php 
                            echo $filterCountry ? '&country='.htmlspecialchars($filterCountry) : ''; 
                            echo $filterIp ? '&ip='.htmlspecialchars($filterIp) : ''; 
                            echo $filterDate ? '&date='.htmlspecialchars($filterDate) : ''; 
                            echo $filterEmail ? '&email='.htmlspecialchars($filterEmail) : ''; 
                        ?>" class="btn btn-secondary">
                            Next <i class="fas fa-chevron-right"></i>
                        </a>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <div class="footer">
        &copy; <?php echo date("Y"); ?> WebTigersAI. All rights reserved.
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/chart.js@3.7.1/dist/chart.min.js"></script>
    <script>
        // Auto-refresh the page every 30 seconds (only if not on filtered/paginated view)
        document.addEventListener('DOMContentLoaded', function() {
            <?php if(!$filterCountry && !$filterIp && !$filterDate && !$filterEmail && $currentPage === 1): ?>
            setTimeout(function() {
                window.location.reload();
            }, 30000);
            <?php endif; ?>
            
            // Show filter form if there are active filters
            <?php if($filterCountry || $filterIp || $filterDate || $filterEmail): ?>
                document.getElementById('filterForm').classList.add('active');
            <?php endif; ?>
            
            // Add confirmation for large exports
            const exportLink = document.querySelector('a[href*="export=1"]');
            if (exportLink) {
                exportLink.addEventListener('click', function(e) {
                    if (<?php echo $totalOpens; ?> > 1000) {
                        if (!confirm('You are about to export approximately <?php echo number_format($totalOpens); ?> records. This may take a while. Continue?')) {
                            e.preventDefault();
                        }
                    }
                });
            }
            
            // Initialize charts
            const timelineCtx = document.getElementById('timelineChart').getContext('2d');
            const timelineChart = new Chart(timelineCtx, {
                type: 'line',
                data: {
                    labels: <?php echo json_encode($timelineLabels); ?>,
                    datasets: [{
                        label: 'Opens by Hour',
                        data: <?php echo json_encode($timelineValues); ?>,
                        backgroundColor: 'rgba(74, 107, 175, 0.2)',
                        borderColor: 'rgba(74, 107, 175, 1)',
                        borderWidth: 2,
                        tension: 0.4,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            mode: 'index',
                            intersect: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                precision: 0
                            }
                        }
                    }
                }
            });
            
            // Country chart data
            const countryCodes = <?php echo json_encode(array_keys($countries)); ?>;
            const countryCounts = <?php echo json_encode(array_column($countries, 'count')); ?>;
            const countryFlags = <?php echo json_encode(array_column($countries, 'flag')); ?>;
            
            const countryCtx = document.getElementById('countryChart').getContext('2d');
            const countryChart = new Chart(countryCtx, {
                type: 'bar',
                data: {
                    labels: countryCodes,
                    datasets: [{
                        label: 'Opens by Country',
                        data: countryCounts,
                        backgroundColor: 'rgba(52, 152, 219, 0.7)',
                        borderColor: 'rgba(52, 152, 219, 1)',
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const flag = countryFlags[context.dataIndex];
                                    const count = context.parsed.y;
                                    return flag + ' ' + count.toLocaleString() + ' opens';
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                precision: 0
                            }
                        }
                    }
                }
            });
            
            // Add animation to new log entries
            const animateNewEntries = () => {
                const entries = document.querySelectorAll('.log-table tr');
                entries.forEach((entry, index) => {
                    setTimeout(() => {
                        entry.classList.add('animate__animated', 'animate__fadeIn');
                    }, index * 50);
                });
            };
            
            animateNewEntries();
            
            // Add loading indicator for large datasets
            const paginationLinks = document.querySelectorAll('.pagination a');
            paginationLinks.forEach(link => {
                link.addEventListener('click', function() {
                    this.innerHTML = '<span class="loading"></span> Loading...';
                });
            });
        });
    </script>
</body>
</html>